/*:
 * @target MZ
 * @plugindesc Fix v4: PKD_Spine2DPlayer の「Save/Load画面キャンセル復帰でSpineが消える/表情が戻らない/二重表示」を安定化
 * @author HS
 * @help Place BELOW PKD_Spine2DPlayer.js
 *
 * 変更点（要旨）
 * 1) Save/Load突入時に現在のアニメ状態をスナップショット
 * 2) Scene_Map.start で参照レジストリ再構築 → 直後＆1フレーム後に再適用
 * 3) applySavedata 後にも最終アニメ再適用（ロード/マップ再生成用）
 * 4) setAnimation 時に lastAnimationState を常に配列で保持
 * 5) CreateSAnim をフックし、同IDが生存していれば再利用（重複生成を防止）
 * 6) Scene_Map.start 後に重複個体を安全に除去（storeは消さない）
 */

(() => {
  'use strict';

  // ---------------------------------
  // 内部フラグ & 一時保存
  // ---------------------------------
  window._hs_spine_shouldReapplyOnMapStart = window._hs_spine_shouldReapplyOnMapStart || false;
  window._hs_spine_lastAnimSnapshot = window._hs_spine_lastAnimSnapshot || {};

  function isFileSceneClass(sceneClass) {
    try {
      return !!sceneClass && (
        sceneClass === window.Scene_Save ||
        sceneClass === window.Scene_Load ||
        (typeof window.Scene_File !== 'undefined' && sceneClass.prototype instanceof window.Scene_File)
      );
    } catch (_e) {
      return false;
    }
  }

  // ---------------------------------
  // Utils: Spine列挙 / レジストリ再構築 / アニメ再適用
  // ---------------------------------
  function hsCollectSpines(root, out) {
    if (!root) return;
    try {
      if (typeof PKD_Sprite_SpineAnimation !== 'undefined' && root instanceof PKD_Sprite_SpineAnimation) {
        out.push(root);
      } else if (typeof PKD_Sprite_SpineMapInst !== 'undefined' && root instanceof PKD_Sprite_SpineMapInst) {
        if (typeof root.spline === 'function') out.push(root.spline());
      }
      const ch = root.children;
      if (ch && ch.length) {
        for (const c of ch) hsCollectSpines(c, out);
      }
    } catch (e) { console.warn(e); }
  }

  function hsRebuildSpineRegistry(sceneMap) {
    try {
      if (!$gameTemp) return;
      const ss = sceneMap && sceneMap._spriteset;
      if (!ss) return;
      const list = [];
      hsCollectSpines(ss, list);
      if (typeof $gameTemp.pAddSpineAnimationSprite === 'function') {
        for (const s of list) {
          if (s) $gameTemp.pAddSpineAnimationSprite(s);
        }
      }
    } catch (e) { console.warn(e); }
  }

  function hsGetCurrentAnimArray(s) {
    try {
      if (s && typeof s.getAnimationState === 'function') {
        const st = s.getAnimationState();
        if (st && st.tracks && st.tracks.length > 0) {
          const entry = st.tracks[0];
          if (entry && entry.animation) {
            const name = entry.animation.name;
            const loop = !!entry.loop;
            const mix  = entry.mixDuration || 0;
            const idx  = (typeof entry.trackIndex === 'number') ? entry.trackIndex : 0;
            if (name) return [name, loop, mix, idx];
          }
        }
      }
    } catch (e) { console.warn(e); }
    return Array.isArray(s && s.lastAnimationState) ? s.lastAnimationState : null;
  }

  function hsSnapshotAllSpineAnimsFromMapScene(sceneMap) {
    try {
      window._hs_spine_lastAnimSnapshot = {};
      const ss = sceneMap && sceneMap._spriteset;
      if (!ss) return;
      const list = [];
      hsCollectSpines(ss, list);
      for (const s of list) {
        try {
          const arr = hsGetCurrentAnimArray(s);
          const skin = (s && typeof s._lastSkinName !== 'undefined') ? s._lastSkinName : null;
          if (s && s.id) {
            if (Array.isArray(arr) && arr[0]) {
              window._hs_spine_lastAnimSnapshot[s.id] = { anim: arr, skin: skin };
            } else if (skin) {
              window._hs_spine_lastAnimSnapshot[s.id] = { anim: null, skin: skin };
            }
          }
        } catch (e) { console.warn(e); }
      }
    } catch (e) { console.warn(e); }
  }

  function hsReapplyLastAnimationState(sceneMap, preferSnapshot) {
    try {
      const ss = sceneMap && sceneMap._spriteset;
      if (!ss) return;
      const list = [];
      hsCollectSpines(ss, list);
      for (const s of list) {
        try {
          let arr = null;
          let skin = null;
          if (preferSnapshot && s && s.id && window._hs_spine_lastAnimSnapshot) {
            const entry = window._hs_spine_lastAnimSnapshot[s.id] || null;
            if (entry) {
              if (Array.isArray(entry)) {
                arr = entry;
              } else if (typeof entry === 'object') {
                arr = entry.anim || null;
                skin = entry.skin || null;
              }
            }
          }
          if (!arr) arr = hsGetCurrentAnimArray(s);
          if (!skin && s && typeof s._lastSkinName !== 'undefined') skin = s._lastSkinName;
          // skin -> animation の順で適用
          if (skin && typeof s.setSkin === 'function') {
            try { s.setSkin(skin, false); } catch (e) { console.warn(e); }
          }
          if (Array.isArray(arr) && arr[0]) {
            s.setAnimation.apply(s, arr);
            s.lastAnimationState = arr.slice();
          }
        } catch (e) { console.warn(e); }
      }
    } catch (e) { console.warn(e); }
  }

  // ---------------------------------
  // デデュープ（二重表示の除去）
  // ---------------------------------
  function hsDedup(sceneMap) {
    try {
      if (!$gameTemp) return;
      const reg = $gameTemp._pSpineAnimations || {};
      const ss = sceneMap && sceneMap._spriteset;
      if (!ss) return;

      const list = [];
      hsCollectSpines(ss, list);

      const byId = new Map();
      for (const s of list) {
        if (!s || !s.id) continue;
        if (!byId.has(s.id)) byId.set(s.id, []);
        byId.get(s.id).push(s);
      }

      for (const [id, arr] of byId.entries()) {
        if (arr.length <= 1) continue;

        // どれを残すか：
        // 1) レジストリ指している個体で可視(parent有)
        // 2) 可視個体
        // 3) レジストリ個体
        // 4) 先頭
        let preferred = reg[id] || null;
        let keep =
          (preferred && preferred.parent) ? preferred :
          (arr.find(s => s && s.parent) || preferred || arr[0]);

        // レジストリが可視以外を指しているなら、可視の keep に差し替え
        if (keep && reg[id] !== keep) {
          try { reg[id] = keep; } catch (e) { console.warn(e); }
        }

        for (const s of arr) {
          if (s === keep) continue;
          try {
            if (s.bindMethod === "bindToMap") {
              const wrapper = s.parent ? s.parent.parent : null;
              if (wrapper && wrapper.parent) wrapper.parent.removeChild(wrapper);
            } else if (s.parent) {
              s.parent.removeChild(s);
            }
            // store（セーブ用の保存オブジェクト）は消さない（removeは呼ばない）
          } catch (e) { console.warn(e); }
        }
      }
    } catch (e) { console.warn(e); }
  }

  // ---------------------------------
  // Save/Load突入検知：push時にスナップショット
  // ---------------------------------

  // SceneManager.push / pop フック
  if (window.SceneManager) {
    const _SceneManager_push = SceneManager.push;
    SceneManager.push = function(sceneClass) {
      try {
        if (isFileSceneClass(sceneClass)) {
          window._hs_spine_shouldReapplyOnMapStart = true;
          const scene = SceneManager._scene;
          if (scene && scene instanceof Scene_Map) {
            hsSnapshotAllSpineAnimsFromMapScene(scene);
          } else {
            window._hs_spine_lastAnimSnapshot = {};
          }
        }
      } catch (_e) {}
      return _SceneManager_push.apply(this, arguments);
    };

    const _SceneManager_pop = SceneManager.pop;
    SceneManager.pop = function() {
      const prev = this._scene;
      const result = _SceneManager_pop.apply(this, arguments);
      const cur = this._scene;
      const isPrevFile =
        (typeof window.Scene_File !== 'undefined' && prev instanceof window.Scene_File) ||
        (typeof window.Scene_Save  !== 'undefined' && prev instanceof window.Scene_Save)  ||
        (typeof window.Scene_Load  !== 'undefined' && prev instanceof window.Scene_Load);
      if (isPrevFile && cur && cur instanceof Scene_Map) {
        function reapply() {
          try {
            hsRebuildSpineRegistry(cur);
            hsReapplyLastAnimationState(cur, true);
            hsDedup(cur);
            window._hs_spine_shouldReapplyOnMapStart = false;
          } catch (e) { console.warn(e); }
        }
        try { reapply(); } catch (e) {}
        setTimeout(function(){ try { reapply(); } catch(e){} }, 0);
        setTimeout(function(){ try { reapply(); } catch(e){} }, 16);
      }
      return result;
    };
  }

  // ---------------------------------
  // Scene_Map.start：復帰直後に再登録 → 表情再適用 → デデュープ
  // （即時＆1フレーム後の二段）
  // ---------------------------------
  if (window.Scene_Map) {
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
      _Scene_Map_start.call(this, ...arguments);

      hsRebuildSpineRegistry(this);

      if (window._hs_spine_shouldReapplyOnMapStart) {
        hsReapplyLastAnimationState(this, true);
        hsDedup(this);

        setTimeout(() => {
          try {
            hsReapplyLastAnimationState(this, true);
            hsDedup(this);
          } catch (e) { console.warn(e); }
        }, 0);

        window._hs_spine_shouldReapplyOnMapStart = false;
      }
    };
  }

  // ---------------------------------
  // ロード/マップ生成時：applySavedata後にも最終アニメ再適用
  // ---------------------------------
  if (window.PKD_Sprite_SpineAnimation) {
    const _applySavedata = PKD_Sprite_SpineAnimation.prototype.applySavedata;
    PKD_Sprite_SpineAnimation.prototype.applySavedata = function(saveObject) {
      const r = _applySavedata.call(this, saveObject);
      try {
        if (saveObject && Array.isArray(saveObject.lastAnimationState)) {
          const arr = saveObject.lastAnimationState;
          if (arr && arr[0]) {
            this.setAnimation(...arr);
            this.lastAnimationState = arr.slice();
          }
        }
      } catch (e) { console.warn(e); }
      return r || this;
    };

    // setAnimation で lastAnimationState を常に更新
    const _setAnimation = PKD_Sprite_SpineAnimation.prototype.setAnimation;
    PKD_Sprite_SpineAnimation.prototype.setAnimation = function(name = "", isLoop = false, mixDuration = 0, trackIndex = 0) {
      const ret = _setAnimation.call(this, name, isLoop, mixDuration, trackIndex);
      try {
        if (name) this.lastAnimationState = [name, isLoop, mixDuration, trackIndex];
      } catch (e) { console.warn(e); }
      return ret;
    };
  }

  // ---------------------------------
  // 重複生成の元を断つ：CreateSAnim をガード
  // 既に同IDが存在し可視ならそれを返し、位置だけ更新。新規生成しない。
  // ---------------------------------
  if (window.CreateSAnim && window.SAnim) {
    const _CreateSAnim = window.CreateSAnim;
    window.CreateSAnim = function(id, filename, x = 0, y = 0) {
      try {
        const a = window.SAnim(id);
        // a.isValid() は filename が空でない時のみ true（Empty対策）
        if (a && typeof a.isValid === 'function' && a.isValid() && a.parent) {
          if (typeof a.setPosition === 'function') a.setPosition(x, y);
          return a;
        }
      } catch (e) { console.warn(e); }
      return _CreateSAnim.call(this, id, filename, x, y);
    };
  }

})();




